'use strict';

var anatomy$1 = require('@zag-js/anatomy');
var date = require('@internationalized/date');
var dateUtils = require('@zag-js/date-utils');
var domQuery = require('@zag-js/dom-query');
var popper = require('@zag-js/popper');
var utils = require('@zag-js/utils');
var core = require('@zag-js/core');
var dismissable = require('@zag-js/dismissable');
var liveRegion = require('@zag-js/live-region');
var types = require('@zag-js/types');

// src/date-picker.anatomy.ts
var anatomy = anatomy$1.createAnatomy("date-picker").parts(
  "clearTrigger",
  "content",
  "control",
  "input",
  "label",
  "monthSelect",
  "nextTrigger",
  "positioner",
  "presetTrigger",
  "prevTrigger",
  "rangeText",
  "root",
  "table",
  "tableBody",
  "tableCell",
  "tableCellTrigger",
  "tableHead",
  "tableHeader",
  "tableRow",
  "trigger",
  "view",
  "viewControl",
  "viewTrigger",
  "yearSelect"
);
var parts = anatomy.build();
var getLabelId = (ctx, index) => ctx.ids?.label?.(index) ?? `datepicker:${ctx.id}:label:${index}`;
var getRootId = (ctx) => ctx.ids?.root ?? `datepicker:${ctx.id}`;
var getTableId = (ctx, id) => ctx.ids?.table?.(id) ?? `datepicker:${ctx.id}:table:${id}`;
var getContentId = (ctx) => ctx.ids?.content ?? `datepicker:${ctx.id}:content`;
var getCellTriggerId = (ctx, id) => ctx.ids?.cellTrigger?.(id) ?? `datepicker:${ctx.id}:cell-trigger:${id}`;
var getPrevTriggerId = (ctx, view) => ctx.ids?.prevTrigger?.(view) ?? `datepicker:${ctx.id}:prev:${view}`;
var getNextTriggerId = (ctx, view) => ctx.ids?.nextTrigger?.(view) ?? `datepicker:${ctx.id}:next:${view}`;
var getViewTriggerId = (ctx, view) => ctx.ids?.viewTrigger?.(view) ?? `datepicker:${ctx.id}:view:${view}`;
var getClearTriggerId = (ctx) => ctx.ids?.clearTrigger ?? `datepicker:${ctx.id}:clear`;
var getControlId = (ctx) => ctx.ids?.control ?? `datepicker:${ctx.id}:control`;
var getInputId = (ctx, index) => ctx.ids?.input?.(index) ?? `datepicker:${ctx.id}:input:${index}`;
var getTriggerId = (ctx) => ctx.ids?.trigger ?? `datepicker:${ctx.id}:trigger`;
var getPositionerId = (ctx) => ctx.ids?.positioner ?? `datepicker:${ctx.id}:positioner`;
var getMonthSelectId = (ctx) => ctx.ids?.monthSelect ?? `datepicker:${ctx.id}:month-select`;
var getYearSelectId = (ctx) => ctx.ids?.yearSelect ?? `datepicker:${ctx.id}:year-select`;
var getFocusedCell = (ctx, view) => domQuery.query(getContentEl(ctx), `[data-part=table-cell-trigger][data-view=${view}][data-focus]:not([data-outside-range])`);
var getTriggerEl = (ctx) => ctx.getById(getTriggerId(ctx));
var getContentEl = (ctx) => ctx.getById(getContentId(ctx));
var getInputEls = (ctx) => domQuery.queryAll(getControlEl(ctx), `[data-part=input]`);
var getYearSelectEl = (ctx) => ctx.getById(getYearSelectId(ctx));
var getMonthSelectEl = (ctx) => ctx.getById(getMonthSelectId(ctx));
var getClearTriggerEl = (ctx) => ctx.getById(getClearTriggerId(ctx));
var getPositionerEl = (ctx) => ctx.getById(getPositionerId(ctx));
var getControlEl = (ctx) => ctx.getById(getControlId(ctx));
function adjustStartAndEndDate(value) {
  const [startDate, endDate] = value;
  if (!startDate || !endDate) return value;
  return startDate.compare(endDate) <= 0 ? value : [endDate, startDate];
}
function isDateWithinRange(date, value) {
  const [startDate, endDate] = value;
  if (!startDate || !endDate) return false;
  return startDate.compare(date) <= 0 && endDate.compare(date) >= 0;
}
function sortDates(values) {
  return values.slice().sort((a, b) => a.compare(b));
}
function getRoleDescription(view) {
  return utils.match(view, {
    year: "calendar decade",
    month: "calendar year",
    day: "calendar month"
  });
}
var PLACEHOLDERS = {
  day: "dd",
  month: "mm",
  year: "yyyy"
};
function getInputPlaceholder(locale) {
  return new date.DateFormatter(locale).formatToParts(/* @__PURE__ */ new Date()).map((item) => PLACEHOLDERS[item.type] ?? item.value).join("");
}
var isValidCharacter = (char, separator) => {
  if (!char) return true;
  return /\d/.test(char) || char === separator || char.length !== 1;
};
var isValidDate = (value) => {
  return !Number.isNaN(value.day) && !Number.isNaN(value.month) && !Number.isNaN(value.year);
};
var ensureValidCharacters = (value, separator) => {
  return value.split("").filter((char) => isValidCharacter(char, separator)).join("");
};
function getLocaleSeparator(locale) {
  const dateFormatter = new Intl.DateTimeFormat(locale);
  const parts2 = dateFormatter.formatToParts(/* @__PURE__ */ new Date());
  const literalPart = parts2.find((part) => part.type === "literal");
  return literalPart ? literalPart.value : "/";
}
var defaultTranslations = {
  dayCell(state) {
    if (state.unavailable) return `Not available. ${state.formattedDate}`;
    if (state.selected) return `Selected date. ${state.formattedDate}`;
    return `Choose ${state.formattedDate}`;
  },
  trigger(open) {
    return open ? "Close calendar" : "Open calendar";
  },
  viewTrigger(view) {
    return utils.match(view, {
      year: "Switch to month view",
      month: "Switch to day view",
      day: "Switch to year view"
    });
  },
  presetTrigger(value) {
    return Array.isArray(value) ? `select ${value[0].toString()} to ${value[1].toString()}` : `select ${value}`;
  },
  prevTrigger(view) {
    return utils.match(view, {
      year: "Switch to previous decade",
      month: "Switch to previous year",
      day: "Switch to previous month"
    });
  },
  nextTrigger(view) {
    return utils.match(view, {
      year: "Switch to next decade",
      month: "Switch to next year",
      day: "Switch to next month"
    });
  },
  // TODO: Revisit this
  placeholder() {
    return { day: "dd", month: "mm", year: "yyyy" };
  },
  content: "calendar",
  monthSelect: "Select month",
  yearSelect: "Select year",
  clearTrigger: "Clear selected dates"
};
function viewToNumber(view, fallback) {
  if (!view) return fallback || 0;
  return view === "day" ? 0 : view === "month" ? 1 : 2;
}
function viewNumberToView(viewNumber) {
  return viewNumber === 0 ? "day" : viewNumber === 1 ? "month" : "year";
}
function clampView(view, minView, maxView) {
  return viewNumberToView(
    utils.clampValue(viewToNumber(view, 0), viewToNumber(minView, 0), viewToNumber(maxView, 2))
  );
}
function isAboveMinView(view, minView) {
  return viewToNumber(view, 0) > viewToNumber(minView, 0);
}
function isBelowMinView(view, minView) {
  return viewToNumber(view, 0) < viewToNumber(minView, 0);
}
function getNextView(view, minView, maxView) {
  const nextViewNumber = viewToNumber(view, 0) + 1;
  return clampView(viewNumberToView(nextViewNumber), minView, maxView);
}
function getPreviousView(view, minView, maxView) {
  const prevViewNumber = viewToNumber(view, 0) - 1;
  return clampView(viewNumberToView(prevViewNumber), minView, maxView);
}
var views = ["day", "month", "year"];
function eachView(cb) {
  views.forEach((view) => cb(view));
}

// src/date-picker.connect.ts
function connect(service, normalize) {
  const { state, context, prop, send, computed, scope } = service;
  const startValue = context.get("startValue");
  const endValue = computed("endValue");
  const selectedValue = context.get("value");
  const focusedValue = context.get("focusedValue");
  const hoveredValue = context.get("hoveredValue");
  const hoveredRangeValue = hoveredValue ? adjustStartAndEndDate([selectedValue[0], hoveredValue]) : [];
  const disabled = prop("disabled");
  const readOnly = prop("readOnly");
  const interactive = computed("isInteractive");
  const min = prop("min");
  const max = prop("max");
  const locale = prop("locale");
  const timeZone = prop("timeZone");
  const startOfWeek = prop("startOfWeek");
  const focused = state.matches("focused");
  const open = state.matches("open");
  const isRangePicker = prop("selectionMode") === "range";
  const isDateUnavailableFn = prop("isDateUnavailable");
  const currentPlacement = context.get("currentPlacement");
  const popperStyles = popper.getPlacementStyles({
    ...prop("positioning"),
    placement: currentPlacement
  });
  const separator = getLocaleSeparator(locale);
  const translations = { ...defaultTranslations, ...prop("translations") };
  function getMonthWeeks(from = startValue) {
    const numOfWeeks = prop("fixedWeeks") ? 6 : void 0;
    return dateUtils.getMonthDays(from, locale, numOfWeeks, startOfWeek);
  }
  function getMonths(props2 = {}) {
    const { format } = props2;
    return dateUtils.getMonthNames(locale, format).map((label, index) => ({ label, value: index + 1 }));
  }
  function getYears() {
    const range = dateUtils.getYearsRange({ from: min?.year ?? 1900, to: max?.year ?? 2100 });
    return range.map((year) => ({ label: year.toString(), value: year }));
  }
  function getDecadeYears(year) {
    const range = dateUtils.getDecadeRange(focusedValue.year);
    return range.map((year2) => ({ label: year2.toString(), value: year2 }));
  }
  function isUnavailable(date) {
    return dateUtils.isDateUnavailable(date, isDateUnavailableFn, locale, min, max);
  }
  function focusMonth(month) {
    const date = startValue ?? dateUtils.getTodayDate(timeZone);
    send({ type: "FOCUS.SET", value: date.set({ month }) });
  }
  function focusYear(year) {
    const date = startValue ?? dateUtils.getTodayDate(timeZone);
    send({ type: "FOCUS.SET", value: date.set({ year }) });
  }
  function getYearTableCellState(props2) {
    const { value, disabled: disabled2 } = props2;
    const cellState = {
      focused: focusedValue.year === props2.value,
      selectable: utils.isValueWithinRange(value, min?.year ?? 0, max?.year ?? 9999),
      selected: !!selectedValue.find((date) => date.year === value),
      valueText: value.toString(),
      get disabled() {
        return disabled2 || !cellState.selectable;
      }
    };
    return cellState;
  }
  function getMonthTableCellState(props2) {
    const { value, disabled: disabled2 } = props2;
    const normalized = focusedValue.set({ month: value });
    const formatter = dateUtils.getMonthFormatter(locale, timeZone);
    const cellState = {
      focused: focusedValue.month === props2.value,
      selectable: !dateUtils.isDateOutsideRange(normalized, min, max),
      selected: !!selectedValue.find((date) => date.month === value && date.year === focusedValue.year),
      valueText: formatter.format(normalized.toDate(timeZone)),
      get disabled() {
        return disabled2 || !cellState.selectable;
      }
    };
    return cellState;
  }
  function getDayTableCellState(props2) {
    const { value, disabled: disabled2, visibleRange = computed("visibleRange") } = props2;
    const formatter = dateUtils.getDayFormatter(locale, timeZone);
    const unitDuration = dateUtils.getUnitDuration(computed("visibleDuration"));
    const end = visibleRange.start.add(unitDuration).subtract({ days: 1 });
    const cellState = {
      invalid: dateUtils.isDateOutsideRange(value, min, max),
      disabled: disabled2 || dateUtils.isDateOutsideRange(value, visibleRange.start, end) || dateUtils.isDateOutsideRange(value, min, max),
      selected: selectedValue.some((date) => dateUtils.isDateEqual(value, date)),
      unavailable: dateUtils.isDateUnavailable(value, isDateUnavailableFn, locale, min, max) && !disabled2,
      outsideRange: dateUtils.isDateOutsideRange(value, visibleRange.start, end),
      inRange: isRangePicker && (isDateWithinRange(value, selectedValue) || isDateWithinRange(value, hoveredRangeValue)),
      firstInRange: isRangePicker && dateUtils.isDateEqual(value, selectedValue[0]),
      lastInRange: isRangePicker && dateUtils.isDateEqual(value, selectedValue[1]),
      today: date.isToday(value, timeZone),
      weekend: date.isWeekend(value, locale),
      formattedDate: formatter.format(value.toDate(timeZone)),
      get focused() {
        return dateUtils.isDateEqual(value, focusedValue) && !cellState.outsideRange;
      },
      get ariaLabel() {
        return translations.dayCell(cellState);
      },
      get selectable() {
        return !cellState.disabled && !cellState.unavailable;
      }
    };
    return cellState;
  }
  function getTableId2(props2) {
    const { view = "day", id } = props2;
    return [view, id].filter(Boolean).join(" ");
  }
  return {
    focused,
    open,
    view: context.get("view"),
    getRangePresetValue(preset) {
      return dateUtils.getDateRangePreset(preset, locale, timeZone);
    },
    getDaysInWeek(week, from = startValue) {
      return dateUtils.getDaysInWeek(week, from, locale, startOfWeek);
    },
    getOffset(duration) {
      const from = startValue.add(duration);
      const end = endValue.add(duration);
      const formatter = dateUtils.getMonthFormatter(locale, timeZone);
      return {
        visibleRange: { start: from, end },
        weeks: getMonthWeeks(from),
        visibleRangeText: {
          start: formatter.format(from.toDate(timeZone)),
          end: formatter.format(end.toDate(timeZone))
        }
      };
    },
    getMonthWeeks,
    isUnavailable,
    weeks: getMonthWeeks(),
    weekDays: dateUtils.getWeekDays(dateUtils.getTodayDate(timeZone), startOfWeek, timeZone, locale),
    visibleRangeText: computed("visibleRangeText"),
    value: selectedValue,
    valueAsDate: selectedValue.map((date) => date.toDate(timeZone)),
    valueAsString: computed("valueAsString"),
    focusedValue,
    focusedValueAsDate: focusedValue?.toDate(timeZone),
    focusedValueAsString: prop("format")(focusedValue, { locale, timeZone }),
    visibleRange: computed("visibleRange"),
    selectToday() {
      const value = dateUtils.constrainValue(dateUtils.getTodayDate(timeZone), min, max);
      send({ type: "VALUE.SET", value });
    },
    setValue(values) {
      const computedValue = values.map((date) => dateUtils.constrainValue(date, min, max));
      send({ type: "VALUE.SET", value: computedValue });
    },
    clearValue() {
      send({ type: "VALUE.CLEAR" });
    },
    setFocusedValue(value) {
      send({ type: "FOCUS.SET", value });
    },
    setOpen(nextOpen) {
      const open2 = state.matches("open");
      if (open2 === nextOpen) return;
      send({ type: nextOpen ? "OPEN" : "CLOSE" });
    },
    focusMonth,
    focusYear,
    getYears,
    getMonths,
    getYearsGrid(props2 = {}) {
      const { columns = 1 } = props2;
      return utils.chunk(getDecadeYears(), columns);
    },
    getDecade() {
      const years = dateUtils.getDecadeRange(focusedValue.year);
      return { start: years.at(0), end: years.at(-1) };
    },
    getMonthsGrid(props2 = {}) {
      const { columns = 1, format } = props2;
      return utils.chunk(getMonths({ format }), columns);
    },
    format(value, opts = { month: "long", year: "numeric" }) {
      return new date.DateFormatter(locale, opts).format(value.toDate(timeZone));
    },
    setView(view) {
      send({ type: "VIEW.SET", view });
    },
    goToNext() {
      send({ type: "GOTO.NEXT", view: context.get("view") });
    },
    goToPrev() {
      send({ type: "GOTO.PREV", view: context.get("view") });
    },
    getRootProps() {
      return normalize.element({
        ...parts.root.attrs,
        dir: prop("dir"),
        id: getRootId(scope),
        "data-state": open ? "open" : "closed",
        "data-disabled": domQuery.dataAttr(disabled),
        "data-readonly": domQuery.dataAttr(readOnly)
      });
    },
    getLabelProps(props2 = {}) {
      const { index = 0 } = props2;
      return normalize.label({
        ...parts.label.attrs,
        id: getLabelId(scope, index),
        dir: prop("dir"),
        htmlFor: getInputId(scope, index),
        "data-state": open ? "open" : "closed",
        "data-index": index,
        "data-disabled": domQuery.dataAttr(disabled),
        "data-readonly": domQuery.dataAttr(readOnly)
      });
    },
    getControlProps() {
      return normalize.element({
        ...parts.control.attrs,
        dir: prop("dir"),
        id: getControlId(scope),
        "data-disabled": domQuery.dataAttr(disabled)
      });
    },
    getRangeTextProps() {
      return normalize.element({
        ...parts.rangeText.attrs,
        dir: prop("dir")
      });
    },
    getContentProps() {
      return normalize.element({
        ...parts.content.attrs,
        hidden: !open,
        dir: prop("dir"),
        "data-state": open ? "open" : "closed",
        "data-placement": currentPlacement,
        id: getContentId(scope),
        tabIndex: -1,
        role: "application",
        "aria-roledescription": "datepicker",
        "aria-label": translations.content
      });
    },
    getTableProps(props2 = {}) {
      const { view = "day", columns = view === "day" ? 7 : 4 } = props2;
      const uid = getTableId2(props2);
      return normalize.element({
        ...parts.table.attrs,
        role: "grid",
        "data-columns": columns,
        "aria-roledescription": getRoleDescription(view),
        id: getTableId(scope, uid),
        "aria-readonly": domQuery.ariaAttr(readOnly),
        "aria-disabled": domQuery.ariaAttr(disabled),
        "aria-multiselectable": domQuery.ariaAttr(prop("selectionMode") !== "single"),
        "data-view": view,
        dir: prop("dir"),
        tabIndex: -1,
        onKeyDown(event) {
          if (event.defaultPrevented) return;
          const keyMap = {
            Enter() {
              if (isUnavailable(focusedValue)) return;
              send({ type: "TABLE.ENTER", view, columns, focus: true });
            },
            ArrowLeft() {
              send({ type: "TABLE.ARROW_LEFT", view, columns, focus: true });
            },
            ArrowRight() {
              send({ type: "TABLE.ARROW_RIGHT", view, columns, focus: true });
            },
            ArrowUp() {
              send({ type: "TABLE.ARROW_UP", view, columns, focus: true });
            },
            ArrowDown() {
              send({ type: "TABLE.ARROW_DOWN", view, columns, focus: true });
            },
            PageUp(event2) {
              send({ type: "TABLE.PAGE_UP", larger: event2.shiftKey, view, columns, focus: true });
            },
            PageDown(event2) {
              send({ type: "TABLE.PAGE_DOWN", larger: event2.shiftKey, view, columns, focus: true });
            },
            Home() {
              send({ type: "TABLE.HOME", view, columns, focus: true });
            },
            End() {
              send({ type: "TABLE.END", view, columns, focus: true });
            }
          };
          const exec = keyMap[domQuery.getEventKey(event, {
            dir: prop("dir")
          })];
          if (exec) {
            exec(event);
            event.preventDefault();
            event.stopPropagation();
          }
        },
        onPointerLeave() {
          send({ type: "TABLE.POINTER_LEAVE" });
        },
        onPointerDown() {
          send({ type: "TABLE.POINTER_DOWN", view });
        },
        onPointerUp() {
          send({ type: "TABLE.POINTER_UP", view });
        }
      });
    },
    getTableHeadProps(props2 = {}) {
      const { view = "day" } = props2;
      return normalize.element({
        ...parts.tableHead.attrs,
        "aria-hidden": true,
        dir: prop("dir"),
        "data-view": view,
        "data-disabled": domQuery.dataAttr(disabled)
      });
    },
    getTableHeaderProps(props2 = {}) {
      const { view = "day" } = props2;
      return normalize.element({
        ...parts.tableHeader.attrs,
        dir: prop("dir"),
        "data-view": view,
        "data-disabled": domQuery.dataAttr(disabled)
      });
    },
    getTableBodyProps(props2 = {}) {
      const { view = "day" } = props2;
      return normalize.element({
        ...parts.tableBody.attrs,
        "data-view": view,
        "data-disabled": domQuery.dataAttr(disabled)
      });
    },
    getTableRowProps(props2 = {}) {
      const { view = "day" } = props2;
      return normalize.element({
        ...parts.tableRow.attrs,
        "aria-disabled": domQuery.ariaAttr(disabled),
        "data-disabled": domQuery.dataAttr(disabled),
        "data-view": view
      });
    },
    getDayTableCellState,
    getDayTableCellProps(props2) {
      const { value } = props2;
      const cellState = getDayTableCellState(props2);
      return normalize.element({
        ...parts.tableCell.attrs,
        role: "gridcell",
        "aria-disabled": domQuery.ariaAttr(!cellState.selectable),
        "aria-selected": cellState.selected || cellState.inRange,
        "aria-invalid": domQuery.ariaAttr(cellState.invalid),
        "aria-current": cellState.today ? "date" : void 0,
        "data-value": value.toString()
      });
    },
    getDayTableCellTriggerProps(props2) {
      const { value } = props2;
      const cellState = getDayTableCellState(props2);
      return normalize.element({
        ...parts.tableCellTrigger.attrs,
        id: getCellTriggerId(scope, value.toString()),
        role: "button",
        dir: prop("dir"),
        tabIndex: cellState.focused ? 0 : -1,
        "aria-label": cellState.ariaLabel,
        "aria-disabled": domQuery.ariaAttr(!cellState.selectable),
        "aria-invalid": domQuery.ariaAttr(cellState.invalid),
        "data-disabled": domQuery.dataAttr(!cellState.selectable),
        "data-selected": domQuery.dataAttr(cellState.selected),
        "data-value": value.toString(),
        "data-view": "day",
        "data-today": domQuery.dataAttr(cellState.today),
        "data-focus": domQuery.dataAttr(cellState.focused),
        "data-unavailable": domQuery.dataAttr(cellState.unavailable),
        "data-range-start": domQuery.dataAttr(cellState.firstInRange),
        "data-range-end": domQuery.dataAttr(cellState.lastInRange),
        "data-in-range": domQuery.dataAttr(cellState.inRange),
        "data-outside-range": domQuery.dataAttr(cellState.outsideRange),
        "data-weekend": domQuery.dataAttr(cellState.weekend),
        onClick(event) {
          if (event.defaultPrevented) return;
          if (!cellState.selectable) return;
          send({ type: "CELL.CLICK", cell: "day", value });
        },
        onPointerMove(event) {
          if (event.pointerType === "touch" || !cellState.selectable) return;
          const focus = event.currentTarget.ownerDocument.activeElement !== event.currentTarget;
          if (hoveredValue && date.isEqualDay(value, hoveredValue)) return;
          send({ type: "CELL.POINTER_MOVE", cell: "day", value, focus });
        }
      });
    },
    getMonthTableCellState,
    getMonthTableCellProps(props2) {
      const { value, columns } = props2;
      const cellState = getMonthTableCellState(props2);
      return normalize.element({
        ...parts.tableCell.attrs,
        dir: prop("dir"),
        colSpan: columns,
        role: "gridcell",
        "aria-selected": domQuery.ariaAttr(cellState.selected),
        "data-selected": domQuery.dataAttr(cellState.selected),
        "aria-disabled": domQuery.ariaAttr(!cellState.selectable),
        "data-value": value
      });
    },
    getMonthTableCellTriggerProps(props2) {
      const { value } = props2;
      const cellState = getMonthTableCellState(props2);
      return normalize.element({
        ...parts.tableCellTrigger.attrs,
        dir: prop("dir"),
        role: "button",
        id: getCellTriggerId(scope, value.toString()),
        "data-selected": domQuery.dataAttr(cellState.selected),
        "aria-disabled": domQuery.ariaAttr(!cellState.selectable),
        "data-disabled": domQuery.dataAttr(!cellState.selectable),
        "data-focus": domQuery.dataAttr(cellState.focused),
        "aria-label": cellState.valueText,
        "data-view": "month",
        "data-value": value,
        tabIndex: cellState.focused ? 0 : -1,
        onClick(event) {
          if (event.defaultPrevented) return;
          if (!cellState.selectable) return;
          send({ type: "CELL.CLICK", cell: "month", value });
        }
      });
    },
    getYearTableCellState,
    getYearTableCellProps(props2) {
      const { value, columns } = props2;
      const cellState = getYearTableCellState(props2);
      return normalize.element({
        ...parts.tableCell.attrs,
        dir: prop("dir"),
        colSpan: columns,
        role: "gridcell",
        "aria-selected": domQuery.ariaAttr(cellState.selected),
        "data-selected": domQuery.dataAttr(cellState.selected),
        "aria-disabled": domQuery.ariaAttr(!cellState.selectable),
        "data-value": value
      });
    },
    getYearTableCellTriggerProps(props2) {
      const { value } = props2;
      const cellState = getYearTableCellState(props2);
      return normalize.element({
        ...parts.tableCellTrigger.attrs,
        dir: prop("dir"),
        role: "button",
        id: getCellTriggerId(scope, value.toString()),
        "data-selected": domQuery.dataAttr(cellState.selected),
        "data-focus": domQuery.dataAttr(cellState.focused),
        "aria-disabled": domQuery.ariaAttr(!cellState.selectable),
        "data-disabled": domQuery.dataAttr(!cellState.selectable),
        "aria-label": cellState.valueText,
        "data-value": value,
        "data-view": "year",
        tabIndex: cellState.focused ? 0 : -1,
        onClick(event) {
          if (event.defaultPrevented) return;
          if (!cellState.selectable) return;
          send({ type: "CELL.CLICK", cell: "year", value });
        }
      });
    },
    getNextTriggerProps(props2 = {}) {
      const { view = "day" } = props2;
      const isDisabled = disabled || !computed("isNextVisibleRangeValid");
      return normalize.button({
        ...parts.nextTrigger.attrs,
        dir: prop("dir"),
        id: getNextTriggerId(scope, view),
        type: "button",
        "aria-label": translations.nextTrigger(view),
        disabled: isDisabled,
        "data-disabled": domQuery.dataAttr(isDisabled),
        onClick(event) {
          if (event.defaultPrevented) return;
          send({ type: "GOTO.NEXT", view });
        }
      });
    },
    getPrevTriggerProps(props2 = {}) {
      const { view = "day" } = props2;
      const isDisabled = disabled || !computed("isPrevVisibleRangeValid");
      return normalize.button({
        ...parts.prevTrigger.attrs,
        dir: prop("dir"),
        id: getPrevTriggerId(scope, view),
        type: "button",
        "aria-label": translations.prevTrigger(view),
        disabled: isDisabled,
        "data-disabled": domQuery.dataAttr(isDisabled),
        onClick(event) {
          if (event.defaultPrevented) return;
          send({ type: "GOTO.PREV", view });
        }
      });
    },
    getClearTriggerProps() {
      return normalize.button({
        ...parts.clearTrigger.attrs,
        id: getClearTriggerId(scope),
        dir: prop("dir"),
        type: "button",
        "aria-label": translations.clearTrigger,
        hidden: !selectedValue.length,
        onClick(event) {
          if (event.defaultPrevented) return;
          send({ type: "VALUE.CLEAR" });
        }
      });
    },
    getTriggerProps() {
      return normalize.button({
        ...parts.trigger.attrs,
        id: getTriggerId(scope),
        dir: prop("dir"),
        type: "button",
        "data-placement": currentPlacement,
        "aria-label": translations.trigger(open),
        "aria-controls": getContentId(scope),
        "data-state": open ? "open" : "closed",
        "aria-haspopup": "grid",
        disabled,
        onClick(event) {
          if (event.defaultPrevented) return;
          if (!interactive) return;
          send({ type: "TRIGGER.CLICK" });
        }
      });
    },
    getViewProps(props2 = {}) {
      const { view = "day" } = props2;
      return normalize.element({
        ...parts.view.attrs,
        "data-view": view,
        hidden: context.get("view") !== view
      });
    },
    getViewTriggerProps(props2 = {}) {
      const { view = "day" } = props2;
      return normalize.button({
        ...parts.viewTrigger.attrs,
        "data-view": view,
        dir: prop("dir"),
        id: getViewTriggerId(scope, view),
        type: "button",
        disabled,
        "aria-label": translations.viewTrigger(view),
        onClick(event) {
          if (event.defaultPrevented) return;
          if (!interactive) return;
          send({ type: "VIEW.TOGGLE", src: "viewTrigger" });
        }
      });
    },
    getViewControlProps(props2 = {}) {
      const { view = "day" } = props2;
      return normalize.element({
        ...parts.viewControl.attrs,
        "data-view": view,
        dir: prop("dir")
      });
    },
    getInputProps(props2 = {}) {
      const { index = 0, fixOnBlur = true } = props2;
      return normalize.input({
        ...parts.input.attrs,
        id: getInputId(scope, index),
        autoComplete: "off",
        autoCorrect: "off",
        spellCheck: "false",
        dir: prop("dir"),
        name: prop("name"),
        "data-index": index,
        "data-state": open ? "open" : "closed",
        readOnly,
        disabled,
        placeholder: prop("placeholder") || getInputPlaceholder(locale),
        defaultValue: computed("valueAsString")[index],
        onBeforeInput(event) {
          const { data } = domQuery.getNativeEvent(event);
          if (!isValidCharacter(data, separator)) {
            event.preventDefault();
          }
        },
        onFocus() {
          send({ type: "INPUT.FOCUS", index });
        },
        onBlur(event) {
          const value = event.currentTarget.value.trim();
          send({ type: "INPUT.BLUR", value, index, fixOnBlur });
        },
        onKeyDown(event) {
          if (event.defaultPrevented) return;
          if (!interactive) return;
          const keyMap = {
            Enter(event2) {
              if (domQuery.isComposingEvent(event2)) return;
              if (isUnavailable(focusedValue)) return;
              if (event2.currentTarget.value.trim() === "") return;
              send({ type: "INPUT.ENTER", value: event2.currentTarget.value, index });
            }
          };
          const exec = keyMap[event.key];
          if (exec) {
            exec(event);
            event.preventDefault();
          }
        },
        onInput(event) {
          const value = event.currentTarget.value;
          send({ type: "INPUT.CHANGE", value: ensureValidCharacters(value, separator), index });
        }
      });
    },
    getMonthSelectProps() {
      return normalize.select({
        ...parts.monthSelect.attrs,
        id: getMonthSelectId(scope),
        "aria-label": translations.monthSelect,
        disabled,
        dir: prop("dir"),
        defaultValue: startValue.month,
        onChange(event) {
          focusMonth(Number(event.currentTarget.value));
        }
      });
    },
    getYearSelectProps() {
      return normalize.select({
        ...parts.yearSelect.attrs,
        id: getYearSelectId(scope),
        disabled,
        "aria-label": translations.yearSelect,
        dir: prop("dir"),
        defaultValue: startValue.year,
        onChange(event) {
          focusYear(Number(event.currentTarget.value));
        }
      });
    },
    getPositionerProps() {
      return normalize.element({
        id: getPositionerId(scope),
        ...parts.positioner.attrs,
        dir: prop("dir"),
        style: popperStyles.floating
      });
    },
    getPresetTriggerProps(props2) {
      const value = Array.isArray(props2.value) ? props2.value : dateUtils.getDateRangePreset(props2.value, locale, timeZone);
      const valueAsString = value.map((item) => item.toDate(timeZone).toDateString());
      return normalize.button({
        ...parts.presetTrigger.attrs,
        "aria-label": translations.presetTrigger(valueAsString),
        type: "button",
        onClick(event) {
          if (event.defaultPrevented) return;
          send({ type: "PRESET.CLICK", value });
        }
      });
    }
  };
}
var { and } = core.createGuards();
function isDateArrayEqual(a, b) {
  if (a?.length !== b?.length) return false;
  const len = Math.max(a.length, b.length);
  for (let i = 0; i < len; i++) {
    if (!dateUtils.isDateEqual(a[i], b[i])) return false;
  }
  return true;
}
var machine = core.createMachine({
  props({ props: props2 }) {
    const locale = props2.locale || "en-US";
    const timeZone = props2.timeZone || "UTC";
    const selectionMode = props2.selectionMode || "single";
    const numOfMonths = props2.numOfMonths || 1;
    const defaultValue = props2.defaultValue ? sortDates(props2.defaultValue).map((date) => dateUtils.constrainValue(date, props2.min, props2.max)) : void 0;
    const value = props2.value ? sortDates(props2.value).map((date) => dateUtils.constrainValue(date, props2.min, props2.max)) : void 0;
    let focusedValue = props2.focusedValue || value?.[0] || defaultValue?.[0] || dateUtils.getTodayDate(timeZone);
    focusedValue = dateUtils.constrainValue(focusedValue, props2.min, props2.max);
    const minView = "day";
    const maxView = "year";
    const defaultView = clampView(props2.view || minView, minView, maxView);
    return {
      locale,
      numOfMonths,
      focusedValue,
      timeZone,
      defaultValue: defaultValue ?? [],
      value,
      selectionMode,
      defaultView,
      minView,
      maxView,
      closeOnSelect: true,
      format(date$1, { locale: locale2, timeZone: timeZone2 }) {
        const formatter = new date.DateFormatter(locale2, { timeZone: timeZone2, day: "2-digit", month: "2-digit", year: "numeric" });
        return formatter.format(date$1.toDate(timeZone2));
      },
      parse(value2, { locale: locale2, timeZone: timeZone2 }) {
        return dateUtils.parseDateString(value2, locale2, timeZone2);
      },
      ...props2,
      positioning: {
        placement: "bottom",
        ...props2.positioning
      }
    };
  },
  initialState({ prop }) {
    const open = prop("open") || prop("defaultOpen");
    return open ? "open" : "idle";
  },
  refs() {
    return {
      announcer: void 0
    };
  },
  context({ prop, bindable, getContext, getComputed }) {
    return {
      focusedValue: bindable(() => ({
        defaultValue: prop("focusedValue"),
        isEqual: dateUtils.isDateEqual,
        hash: (v) => v.toString(),
        sync: true,
        onChange(focusedValue) {
          const context = getContext();
          const computed = getComputed();
          prop("onFocusChange")?.({
            value: context.get("value"),
            valueAsString: computed("valueAsString"),
            view: context.get("view"),
            focusedValue
          });
        }
      })),
      value: bindable(() => ({
        defaultValue: prop("defaultValue"),
        value: prop("value"),
        isEqual: isDateArrayEqual,
        hash: (v) => v.map((date) => date.toString()).join(","),
        onChange(value) {
          const context = getContext();
          const valueAsString = value.map(
            (date) => prop("format")(date, { locale: prop("locale"), timeZone: prop("timeZone") })
          );
          prop("onValueChange")?.({ value, valueAsString, view: context.get("view") });
        }
      })),
      inputValue: bindable(() => ({
        defaultValue: ""
      })),
      activeIndex: bindable(() => ({
        defaultValue: 0,
        sync: true
      })),
      hoveredValue: bindable(() => ({
        defaultValue: null,
        isEqual: (a, b) => b !== null && a !== null && dateUtils.isDateEqual(a, b)
      })),
      view: bindable(() => ({
        defaultValue: prop("defaultView"),
        value: prop("view"),
        onChange(value) {
          prop("onViewChange")?.({ view: value });
        }
      })),
      startValue: bindable(() => {
        return {
          defaultValue: dateUtils.alignDate(prop("focusedValue"), "start", { months: prop("numOfMonths") }, prop("locale")),
          isEqual: dateUtils.isDateEqual
        };
      }),
      currentPlacement: bindable(() => ({
        defaultValue: void 0
      })),
      restoreFocus: bindable(() => ({
        defaultValue: false
      }))
    };
  },
  computed: {
    isInteractive: ({ prop }) => !prop("disabled") && !prop("readOnly"),
    visibleDuration: ({ prop }) => ({ months: prop("numOfMonths") }),
    endValue: ({ context, computed }) => dateUtils.getEndDate(context.get("startValue"), computed("visibleDuration")),
    visibleRange: ({ context, computed }) => ({ start: context.get("startValue"), end: computed("endValue") }),
    visibleRangeText({ context, prop, computed }) {
      const timeZone = prop("timeZone");
      const formatter = new date.DateFormatter(prop("locale"), { month: "long", year: "numeric", timeZone });
      const start = formatter.format(context.get("startValue").toDate(timeZone));
      const end = formatter.format(computed("endValue").toDate(timeZone));
      const formatted = prop("selectionMode") === "range" ? `${start} - ${end}` : start;
      return { start, end, formatted };
    },
    isPrevVisibleRangeValid: ({ context, prop }) => !dateUtils.isPreviousRangeInvalid(context.get("startValue"), prop("min"), prop("max")),
    isNextVisibleRangeValid: ({ prop, computed }) => !dateUtils.isNextRangeInvalid(computed("endValue"), prop("min"), prop("max")),
    valueAsString({ context, prop }) {
      const value = context.get("value");
      return value.map((date) => prop("format")(date, { locale: prop("locale"), timeZone: prop("timeZone") }));
    }
  },
  effects: ["setupLiveRegion"],
  watch({ track, prop, context, action, computed }) {
    track([() => prop("locale")], () => {
      action(["setStartValue"]);
    });
    track([() => context.hash("focusedValue")], () => {
      action([
        "setStartValue",
        "syncMonthSelectElement",
        "syncYearSelectElement",
        "focusActiveCellIfNeeded",
        "setHoveredValueIfKeyboard"
      ]);
    });
    track([() => context.get("inputValue")], () => {
      action(["syncInputValue"]);
    });
    track([() => context.hash("value")], () => {
      action(["syncInputElement"]);
    });
    track([() => computed("valueAsString").toString()], () => {
      action(["announceValueText"]);
    });
    track([() => context.get("view")], () => {
      action(["focusActiveCell"]);
    });
    track([() => prop("open")], () => {
      action(["toggleVisibility"]);
    });
  },
  on: {
    "VALUE.SET": {
      actions: ["setDateValue", "setFocusedDate"]
    },
    "VIEW.SET": {
      actions: ["setView"]
    },
    "FOCUS.SET": {
      actions: ["setFocusedDate"]
    },
    "VALUE.CLEAR": {
      actions: ["clearDateValue", "clearFocusedDate", "focusFirstInputElement"]
    },
    "INPUT.CHANGE": [
      {
        guard: "isInputValueEmpty",
        actions: ["setInputValue", "clearDateValue", "clearFocusedDate"]
      },
      {
        actions: ["setInputValue", "focusParsedDate"]
      }
    ],
    "INPUT.ENTER": {
      actions: ["focusParsedDate", "selectFocusedDate"]
    },
    "INPUT.FOCUS": {
      actions: ["setActiveIndex"]
    },
    "INPUT.BLUR": [
      {
        guard: "shouldFixOnBlur",
        actions: ["setActiveIndexToStart", "selectParsedDate"]
      },
      {
        actions: ["setActiveIndexToStart"]
      }
    ],
    "PRESET.CLICK": [
      {
        guard: "isOpenControlled",
        actions: ["setDateValue", "setFocusedDate", "invokeOnClose"]
      },
      {
        target: "focused",
        actions: ["setDateValue", "setFocusedDate", "focusInputElement"]
      }
    ],
    "GOTO.NEXT": [
      {
        guard: "isYearView",
        actions: ["focusNextDecade", "announceVisibleRange"]
      },
      {
        guard: "isMonthView",
        actions: ["focusNextYear", "announceVisibleRange"]
      },
      {
        actions: ["focusNextPage"]
      }
    ],
    "GOTO.PREV": [
      {
        guard: "isYearView",
        actions: ["focusPreviousDecade", "announceVisibleRange"]
      },
      {
        guard: "isMonthView",
        actions: ["focusPreviousYear", "announceVisibleRange"]
      },
      {
        actions: ["focusPreviousPage"]
      }
    ]
  },
  states: {
    idle: {
      tags: ["closed"],
      on: {
        "CONTROLLED.OPEN": {
          target: "open",
          actions: ["focusFirstSelectedDate", "focusActiveCell"]
        },
        "TRIGGER.CLICK": [
          {
            guard: "isOpenControlled",
            actions: ["invokeOnOpen"]
          },
          {
            target: "open",
            actions: ["focusFirstSelectedDate", "focusActiveCell", "invokeOnOpen"]
          }
        ],
        OPEN: [
          {
            guard: "isOpenControlled",
            actions: ["invokeOnOpen"]
          },
          {
            target: "open",
            actions: ["focusFirstSelectedDate", "focusActiveCell", "invokeOnOpen"]
          }
        ]
      }
    },
    focused: {
      tags: ["closed"],
      on: {
        "CONTROLLED.OPEN": {
          target: "open",
          actions: ["focusFirstSelectedDate", "focusActiveCell"]
        },
        "TRIGGER.CLICK": [
          {
            guard: "isOpenControlled",
            actions: ["invokeOnOpen"]
          },
          {
            target: "open",
            actions: ["focusFirstSelectedDate", "focusActiveCell", "invokeOnOpen"]
          }
        ],
        OPEN: [
          {
            guard: "isOpenControlled",
            actions: ["invokeOnOpen"]
          },
          {
            target: "open",
            actions: ["focusFirstSelectedDate", "focusActiveCell", "invokeOnOpen"]
          }
        ]
      }
    },
    open: {
      tags: ["open"],
      effects: ["trackDismissableElement", "trackPositioning"],
      exit: ["clearHoveredDate", "resetView"],
      on: {
        "CONTROLLED.CLOSE": [
          {
            guard: and("shouldRestoreFocus", "isInteractOutsideEvent"),
            target: "focused",
            actions: ["focusTriggerElement"]
          },
          {
            guard: "shouldRestoreFocus",
            target: "focused",
            actions: ["focusInputElement"]
          },
          {
            target: "idle"
          }
        ],
        "CELL.CLICK": [
          {
            guard: "isAboveMinView",
            actions: ["setFocusedValueForView", "setPreviousView"]
          },
          {
            guard: and("isRangePicker", "hasSelectedRange"),
            actions: ["setActiveIndexToStart", "resetSelection", "setActiveIndexToEnd"]
          },
          // === Grouped transitions (based on `closeOnSelect` and `isOpenControlled`) ===
          {
            guard: and("isRangePicker", "isSelectingEndDate", "closeOnSelect", "isOpenControlled"),
            actions: ["setFocusedDate", "setSelectedDate", "setActiveIndexToStart", "invokeOnClose", "setRestoreFocus"]
          },
          {
            guard: and("isRangePicker", "isSelectingEndDate", "closeOnSelect"),
            target: "focused",
            actions: [
              "setFocusedDate",
              "setSelectedDate",
              "setActiveIndexToStart",
              "invokeOnClose",
              "focusInputElement"
            ]
          },
          {
            guard: and("isRangePicker", "isSelectingEndDate"),
            actions: ["setFocusedDate", "setSelectedDate", "setActiveIndexToStart", "clearHoveredDate"]
          },
          // ===
          {
            guard: "isRangePicker",
            actions: ["setFocusedDate", "setSelectedDate", "setActiveIndexToEnd"]
          },
          {
            guard: "isMultiPicker",
            actions: ["setFocusedDate", "toggleSelectedDate"]
          },
          // === Grouped transitions (based on `closeOnSelect` and `isOpenControlled`) ===
          {
            guard: and("closeOnSelect", "isOpenControlled"),
            actions: ["setFocusedDate", "setSelectedDate", "invokeOnClose"]
          },
          {
            guard: "closeOnSelect",
            target: "focused",
            actions: ["setFocusedDate", "setSelectedDate", "invokeOnClose", "focusInputElement"]
          },
          {
            actions: ["setFocusedDate", "setSelectedDate"]
          }
          // ===
        ],
        "CELL.POINTER_MOVE": {
          guard: and("isRangePicker", "isSelectingEndDate"),
          actions: ["setHoveredDate", "setFocusedDate"]
        },
        "TABLE.POINTER_LEAVE": {
          guard: "isRangePicker",
          actions: ["clearHoveredDate"]
        },
        "TABLE.POINTER_DOWN": {
          actions: ["disableTextSelection"]
        },
        "TABLE.POINTER_UP": {
          actions: ["enableTextSelection"]
        },
        "TABLE.ESCAPE": [
          {
            guard: "isOpenControlled",
            actions: ["focusFirstSelectedDate", "invokeOnClose"]
          },
          {
            target: "focused",
            actions: ["focusFirstSelectedDate", "invokeOnClose", "focusTriggerElement"]
          }
        ],
        "TABLE.ENTER": [
          {
            guard: "isAboveMinView",
            actions: ["setPreviousView"]
          },
          {
            guard: and("isRangePicker", "hasSelectedRange"),
            actions: ["setActiveIndexToStart", "clearDateValue", "setSelectedDate", "setActiveIndexToEnd"]
          },
          // === Grouped transitions (based on `closeOnSelect` and `isOpenControlled`) ===
          {
            guard: and("isRangePicker", "isSelectingEndDate", "closeOnSelect", "isOpenControlled"),
            actions: ["setSelectedDate", "setActiveIndexToStart", "invokeOnClose"]
          },
          {
            guard: and("isRangePicker", "isSelectingEndDate", "closeOnSelect"),
            target: "focused",
            actions: ["setSelectedDate", "setActiveIndexToStart", "invokeOnClose", "focusInputElement"]
          },
          {
            guard: and("isRangePicker", "isSelectingEndDate"),
            actions: ["setSelectedDate", "setActiveIndexToStart"]
          },
          // ===
          {
            guard: "isRangePicker",
            actions: ["setSelectedDate", "setActiveIndexToEnd", "focusNextDay"]
          },
          {
            guard: "isMultiPicker",
            actions: ["toggleSelectedDate"]
          },
          // === Grouped transitions (based on `closeOnSelect` and `isOpenControlled`) ===
          {
            guard: and("closeOnSelect", "isOpenControlled"),
            actions: ["selectFocusedDate", "invokeOnClose"]
          },
          {
            guard: "closeOnSelect",
            target: "focused",
            actions: ["selectFocusedDate", "invokeOnClose", "focusInputElement"]
          },
          {
            actions: ["selectFocusedDate"]
          }
          // ===
        ],
        "TABLE.ARROW_RIGHT": [
          {
            guard: "isMonthView",
            actions: ["focusNextMonth"]
          },
          {
            guard: "isYearView",
            actions: ["focusNextYear"]
          },
          {
            actions: ["focusNextDay", "setHoveredDate"]
          }
        ],
        "TABLE.ARROW_LEFT": [
          {
            guard: "isMonthView",
            actions: ["focusPreviousMonth"]
          },
          {
            guard: "isYearView",
            actions: ["focusPreviousYear"]
          },
          {
            actions: ["focusPreviousDay"]
          }
        ],
        "TABLE.ARROW_UP": [
          {
            guard: "isMonthView",
            actions: ["focusPreviousMonthColumn"]
          },
          {
            guard: "isYearView",
            actions: ["focusPreviousYearColumn"]
          },
          {
            actions: ["focusPreviousWeek"]
          }
        ],
        "TABLE.ARROW_DOWN": [
          {
            guard: "isMonthView",
            actions: ["focusNextMonthColumn"]
          },
          {
            guard: "isYearView",
            actions: ["focusNextYearColumn"]
          },
          {
            actions: ["focusNextWeek"]
          }
        ],
        "TABLE.PAGE_UP": {
          actions: ["focusPreviousSection"]
        },
        "TABLE.PAGE_DOWN": {
          actions: ["focusNextSection"]
        },
        "TABLE.HOME": [
          {
            guard: "isMonthView",
            actions: ["focusFirstMonth"]
          },
          {
            guard: "isYearView",
            actions: ["focusFirstYear"]
          },
          {
            actions: ["focusSectionStart"]
          }
        ],
        "TABLE.END": [
          {
            guard: "isMonthView",
            actions: ["focusLastMonth"]
          },
          {
            guard: "isYearView",
            actions: ["focusLastYear"]
          },
          {
            actions: ["focusSectionEnd"]
          }
        ],
        "TRIGGER.CLICK": [
          {
            guard: "isOpenControlled",
            actions: ["invokeOnClose"]
          },
          {
            target: "focused",
            actions: ["invokeOnClose"]
          }
        ],
        "VIEW.TOGGLE": {
          actions: ["setNextView"]
        },
        INTERACT_OUTSIDE: [
          {
            guard: "isOpenControlled",
            actions: ["setActiveIndexToStart", "invokeOnClose"]
          },
          {
            guard: "shouldRestoreFocus",
            target: "focused",
            actions: ["setActiveIndexToStart", "invokeOnClose", "focusTriggerElement"]
          },
          {
            target: "idle",
            actions: ["setActiveIndexToStart", "invokeOnClose"]
          }
        ],
        CLOSE: [
          {
            guard: "isOpenControlled",
            actions: ["setActiveIndexToStart", "invokeOnClose"]
          },
          {
            target: "idle",
            actions: ["setActiveIndexToStart", "invokeOnClose"]
          }
        ]
      }
    }
  },
  implementations: {
    guards: {
      isAboveMinView: ({ context, prop }) => isAboveMinView(context.get("view"), prop("minView")),
      isDayView: ({ context, event }) => (event.view || context.get("view")) === "day",
      isMonthView: ({ context, event }) => (event.view || context.get("view")) === "month",
      isYearView: ({ context, event }) => (event.view || context.get("view")) === "year",
      isRangePicker: ({ prop }) => prop("selectionMode") === "range",
      hasSelectedRange: ({ context }) => context.get("value").length === 2,
      isMultiPicker: ({ prop }) => prop("selectionMode") === "multiple",
      shouldRestoreFocus: ({ context }) => !!context.get("restoreFocus"),
      isSelectingEndDate: ({ context }) => context.get("activeIndex") === 1,
      closeOnSelect: ({ prop }) => !!prop("closeOnSelect"),
      isOpenControlled: ({ prop }) => prop("open") != void 0,
      isInteractOutsideEvent: ({ event }) => event.previousEvent?.type === "INTERACT_OUTSIDE",
      isInputValueEmpty: ({ event }) => event.value.trim() === "",
      shouldFixOnBlur: ({ event }) => !!event.fixOnBlur
    },
    effects: {
      trackPositioning({ context, prop, scope }) {
        if (!context.get("currentPlacement")) {
          context.set("currentPlacement", prop("positioning").placement);
        }
        const anchorEl = getControlEl(scope);
        const getPositionerEl2 = () => getPositionerEl(scope);
        return popper.getPlacement(anchorEl, getPositionerEl2, {
          ...prop("positioning"),
          defer: true,
          onComplete(data) {
            context.set("currentPlacement", data.placement);
          }
        });
      },
      setupLiveRegion({ scope, refs }) {
        const doc = scope.getDoc();
        refs.set("announcer", liveRegion.createLiveRegion({ level: "assertive", document: doc }));
        return () => refs.get("announcer")?.destroy?.();
      },
      trackDismissableElement({ scope, send, context }) {
        const getContentEl2 = () => getContentEl(scope);
        return dismissable.trackDismissableElement(getContentEl2, {
          defer: true,
          exclude: [...getInputEls(scope), getTriggerEl(scope), getClearTriggerEl(scope)],
          onInteractOutside(event) {
            context.set("restoreFocus", !event.detail.focusable);
          },
          onDismiss() {
            send({ type: "INTERACT_OUTSIDE" });
          },
          onEscapeKeyDown(event) {
            event.preventDefault();
            send({ type: "TABLE.ESCAPE", src: "dismissable" });
          }
        });
      }
    },
    actions: {
      setNextView({ context, prop }) {
        const nextView = getNextView(context.get("view"), prop("minView"), prop("maxView"));
        context.set("view", nextView);
      },
      setPreviousView({ context, prop }) {
        const prevView = getPreviousView(context.get("view"), prop("minView"), prop("maxView"));
        context.set("view", prevView);
      },
      setView({ context, event }) {
        context.set("view", event.view);
      },
      setRestoreFocus({ context }) {
        context.set("restoreFocus", true);
      },
      announceValueText({ context, prop, refs }) {
        const announceText = context.get("value").map((date) => dateUtils.formatSelectedDate(date, null, prop("locale"), prop("timeZone")));
        refs.get("announcer")?.announce(announceText.join(","), 3e3);
      },
      announceVisibleRange({ computed, refs }) {
        const { formatted } = computed("visibleRangeText");
        refs.get("announcer")?.announce(formatted);
      },
      disableTextSelection({ scope }) {
        domQuery.disableTextSelection({ target: getContentEl(scope), doc: scope.getDoc() });
      },
      enableTextSelection({ scope }) {
        domQuery.restoreTextSelection({ doc: scope.getDoc(), target: getContentEl(scope) });
      },
      focusFirstSelectedDate(params) {
        const { context } = params;
        if (!context.get("value").length) return;
        setFocusedValue(params, context.get("value")[0]);
      },
      syncInputElement({ scope, computed }) {
        domQuery.raf(() => {
          const inputEls = getInputEls(scope);
          inputEls.forEach((inputEl, index) => {
            domQuery.setElementValue(inputEl, computed("valueAsString")[index] || "");
          });
        });
      },
      setFocusedDate(params) {
        const { event } = params;
        const value = Array.isArray(event.value) ? event.value[0] : event.value;
        setFocusedValue(params, value);
      },
      setFocusedValueForView(params) {
        const { context, event } = params;
        setFocusedValue(params, context.get("focusedValue").set({ [context.get("view")]: event.value }));
      },
      focusNextMonth(params) {
        const { context } = params;
        setFocusedValue(params, context.get("focusedValue").add({ months: 1 }));
      },
      focusPreviousMonth(params) {
        const { context } = params;
        setFocusedValue(params, context.get("focusedValue").subtract({ months: 1 }));
      },
      setDateValue({ context, event, prop }) {
        if (!Array.isArray(event.value)) return;
        const value = event.value.map((date) => dateUtils.constrainValue(date, prop("min"), prop("max")));
        context.set("value", value);
      },
      clearDateValue({ context }) {
        context.set("value", []);
      },
      setSelectedDate(params) {
        const { context, event } = params;
        const values = Array.from(context.get("value"));
        values[context.get("activeIndex")] = normalizeValue(params, event.value ?? context.get("focusedValue"));
        context.set("value", adjustStartAndEndDate(values));
      },
      resetSelection(params) {
        const { context, event } = params;
        context.set("value", [event.value ?? context.get("focusedValue")]);
      },
      toggleSelectedDate(params) {
        const { context, event } = params;
        const currentValue = normalizeValue(params, event.value ?? context.get("focusedValue"));
        const index = context.get("value").findIndex((date) => dateUtils.isDateEqual(date, currentValue));
        if (index === -1) {
          const values = [...context.get("value"), currentValue];
          context.set("value", sortDates(values));
        } else {
          const values = Array.from(context.get("value"));
          values.splice(index, 1);
          context.set("value", sortDates(values));
        }
      },
      setHoveredDate({ context, event }) {
        context.set("hoveredValue", event.value);
      },
      clearHoveredDate({ context }) {
        context.set("hoveredValue", null);
      },
      selectFocusedDate({ context, computed }) {
        const values = Array.from(context.get("value"));
        const activeIndex = context.get("activeIndex");
        values[activeIndex] = context.get("focusedValue").copy();
        context.set("value", adjustStartAndEndDate(values));
        const valueAsString = computed("valueAsString");
        context.set("inputValue", valueAsString[activeIndex]);
      },
      focusPreviousDay(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").subtract({ days: 1 });
        setFocusedValue(params, nextValue);
      },
      focusNextDay(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").add({ days: 1 });
        setFocusedValue(params, nextValue);
      },
      focusPreviousWeek(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").subtract({ weeks: 1 });
        setFocusedValue(params, nextValue);
      },
      focusNextWeek(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").add({ weeks: 1 });
        setFocusedValue(params, nextValue);
      },
      focusNextPage(params) {
        const { context, computed, prop } = params;
        const nextPage = dateUtils.getNextPage(
          context.get("focusedValue"),
          context.get("startValue"),
          computed("visibleDuration"),
          prop("locale"),
          prop("min"),
          prop("max")
        );
        setAdjustedValue(params, nextPage);
      },
      focusPreviousPage(params) {
        const { context, computed, prop } = params;
        const previousPage = dateUtils.getPreviousPage(
          context.get("focusedValue"),
          context.get("startValue"),
          computed("visibleDuration"),
          prop("locale"),
          prop("min"),
          prop("max")
        );
        setAdjustedValue(params, previousPage);
      },
      focusSectionStart(params) {
        const { context } = params;
        setFocusedValue(params, context.get("startValue").copy());
      },
      focusSectionEnd(params) {
        const { computed } = params;
        setFocusedValue(params, computed("endValue").copy());
      },
      focusNextSection(params) {
        const { context, event, computed, prop } = params;
        const nextSection = dateUtils.getNextSection(
          context.get("focusedValue"),
          context.get("startValue"),
          event.larger,
          computed("visibleDuration"),
          prop("locale"),
          prop("min"),
          prop("max")
        );
        if (!nextSection) return;
        setAdjustedValue(params, nextSection);
      },
      focusPreviousSection(params) {
        const { context, event, computed, prop } = params;
        const previousSection = dateUtils.getPreviousSection(
          context.get("focusedValue"),
          context.get("startValue"),
          event.larger,
          computed("visibleDuration"),
          prop("locale"),
          prop("min"),
          prop("max")
        );
        if (!previousSection) return;
        setAdjustedValue(params, previousSection);
      },
      focusNextYear(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").add({ years: 1 });
        setFocusedValue(params, nextValue);
      },
      focusPreviousYear(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").subtract({ years: 1 });
        setFocusedValue(params, nextValue);
      },
      focusNextDecade(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").add({ years: 10 });
        setFocusedValue(params, nextValue);
      },
      focusPreviousDecade(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").subtract({ years: 10 });
        setFocusedValue(params, nextValue);
      },
      clearFocusedDate(params) {
        const { prop } = params;
        setFocusedValue(params, dateUtils.getTodayDate(prop("timeZone")));
      },
      focusPreviousMonthColumn(params) {
        const { context, event } = params;
        const nextValue = context.get("focusedValue").subtract({ months: event.columns });
        setFocusedValue(params, nextValue);
      },
      focusNextMonthColumn(params) {
        const { context, event } = params;
        const nextValue = context.get("focusedValue").add({ months: event.columns });
        setFocusedValue(params, nextValue);
      },
      focusPreviousYearColumn(params) {
        const { context, event } = params;
        const nextValue = context.get("focusedValue").subtract({ years: event.columns });
        setFocusedValue(params, nextValue);
      },
      focusNextYearColumn(params) {
        const { context, event } = params;
        const nextValue = context.get("focusedValue").add({ years: event.columns });
        setFocusedValue(params, nextValue);
      },
      focusFirstMonth(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").set({ month: 0 });
        setFocusedValue(params, nextValue);
      },
      focusLastMonth(params) {
        const { context } = params;
        const nextValue = context.get("focusedValue").set({ month: 12 });
        setFocusedValue(params, nextValue);
      },
      focusFirstYear(params) {
        const { context } = params;
        const range = dateUtils.getDecadeRange(context.get("focusedValue").year);
        const nextValue = context.get("focusedValue").set({ year: range[0] });
        setFocusedValue(params, nextValue);
      },
      focusLastYear(params) {
        const { context } = params;
        const range = dateUtils.getDecadeRange(context.get("focusedValue").year);
        const nextValue = context.get("focusedValue").set({ year: range[range.length - 1] });
        setFocusedValue(params, nextValue);
      },
      setActiveIndex({ context, event }) {
        context.set("activeIndex", event.index);
      },
      setActiveIndexToEnd({ context }) {
        context.set("activeIndex", 1);
      },
      setActiveIndexToStart({ context }) {
        context.set("activeIndex", 0);
      },
      focusActiveCell({ scope, context }) {
        domQuery.raf(() => {
          const view = context.get("view");
          getFocusedCell(scope, view)?.focus({ preventScroll: true });
        });
      },
      focusActiveCellIfNeeded({ scope, context, event }) {
        if (!event.focus) return;
        domQuery.raf(() => {
          const view = context.get("view");
          getFocusedCell(scope, view)?.focus({ preventScroll: true });
        });
      },
      setHoveredValueIfKeyboard({ context, event, prop }) {
        if (!event.type.startsWith("TABLE.ARROW") || prop("selectionMode") !== "range" || context.get("activeIndex") === 0)
          return;
        context.set("hoveredValue", context.get("focusedValue").copy());
      },
      focusTriggerElement({ scope }) {
        domQuery.raf(() => {
          getTriggerEl(scope)?.focus({ preventScroll: true });
        });
      },
      focusFirstInputElement({ scope }) {
        domQuery.raf(() => {
          const [inputEl] = getInputEls(scope);
          inputEl?.focus({ preventScroll: true });
        });
      },
      focusInputElement({ scope }) {
        domQuery.raf(() => {
          const inputEls = getInputEls(scope);
          const lastIndexWithValue = inputEls.findLastIndex((inputEl2) => inputEl2.value !== "");
          const indexToFocus = Math.max(lastIndexWithValue, 0);
          const inputEl = inputEls[indexToFocus];
          inputEl?.focus({ preventScroll: true });
          inputEl?.setSelectionRange(inputEl.value.length, inputEl.value.length);
        });
      },
      syncMonthSelectElement({ scope, context }) {
        const monthSelectEl = getMonthSelectEl(scope);
        domQuery.setElementValue(monthSelectEl, context.get("startValue").month.toString());
      },
      syncYearSelectElement({ scope, context }) {
        const yearSelectEl = getYearSelectEl(scope);
        domQuery.setElementValue(yearSelectEl, context.get("startValue").year.toString());
      },
      setInputValue({ context, event }) {
        if (context.get("activeIndex") !== event.index) return;
        context.set("inputValue", event.value);
      },
      syncInputValue({ scope, context, event }) {
        queueMicrotask(() => {
          const inputEls = getInputEls(scope);
          const idx = event.index ?? context.get("activeIndex");
          domQuery.setElementValue(inputEls[idx], context.get("inputValue"));
        });
      },
      focusParsedDate(params) {
        const { event, prop } = params;
        if (event.index == null) return;
        const parse2 = prop("parse");
        const date = parse2(event.value, { locale: prop("locale"), timeZone: prop("timeZone") });
        if (!date || !isValidDate(date)) return;
        setFocusedValue(params, date);
      },
      selectParsedDate({ context, event, computed, prop }) {
        if (event.index == null) return;
        const parse2 = prop("parse");
        let date = parse2(event.value, { locale: prop("locale"), timeZone: prop("timeZone") });
        if (!date || !isValidDate(date)) {
          if (event.value) {
            date = context.get("focusedValue").copy();
          }
        }
        if (!date) return;
        const values = Array.from(context.get("value"));
        values[event.index] = date;
        context.set("value", values);
        context.set("inputValue", computed("valueAsString")[event.index]);
      },
      resetView({ context }) {
        context.set("view", context.initial("view"));
      },
      setStartValue({ context, computed, prop }) {
        const focusedValue = context.get("focusedValue");
        const outside = dateUtils.isDateOutsideRange(focusedValue, context.get("startValue"), computed("endValue"));
        if (!outside) return;
        const startValue = dateUtils.alignDate(focusedValue, "start", { months: prop("numOfMonths") }, prop("locale"));
        context.set("startValue", startValue);
      },
      invokeOnOpen({ prop }) {
        prop("onOpenChange")?.({ open: true });
      },
      invokeOnClose({ prop }) {
        prop("onOpenChange")?.({ open: false });
      },
      toggleVisibility({ event, send, prop }) {
        send({ type: prop("open") ? "CONTROLLED.OPEN" : "CONTROLLED.CLOSE", previousEvent: event });
      }
    }
  }
});
var normalizeValue = (ctx, value) => {
  const { context, prop } = ctx;
  const view = context.get("view");
  let dateValue = typeof value === "number" ? context.get("focusedValue").set({ [view]: value }) : value;
  eachView((view2) => {
    if (isBelowMinView(view2, prop("minView"))) {
      dateValue = dateValue.set({ [view2]: view2 === "day" ? 1 : 0 });
    }
  });
  return dateValue;
};
function setFocusedValue(ctx, mixedValue) {
  const { context, prop, computed } = ctx;
  if (!mixedValue) return;
  const value = normalizeValue(ctx, mixedValue);
  if (dateUtils.isDateEqual(context.get("focusedValue"), value)) return;
  const adjustFn = dateUtils.getAdjustedDateFn(computed("visibleDuration"), prop("locale"), prop("min"), prop("max"));
  const adjustedValue = adjustFn({
    focusedDate: value,
    startDate: context.get("startValue")
  });
  context.set("startValue", adjustedValue.startDate);
  context.set("focusedValue", adjustedValue.focusedDate);
}
function setAdjustedValue(ctx, value) {
  const { context } = ctx;
  context.set("startValue", value.startDate);
  const focusedValue = context.get("focusedValue");
  if (dateUtils.isDateEqual(focusedValue, value.focusedDate)) return;
  context.set("focusedValue", value.focusedDate);
}
function parse(value) {
  if (Array.isArray(value)) {
    return value.map((v) => parse(v));
  }
  if (value instanceof Date) {
    return new date.CalendarDate(value.getFullYear(), value.getMonth() + 1, value.getDate());
  }
  return date.parseDate(value);
}
var props = types.createProps()([
  "closeOnSelect",
  "dir",
  "disabled",
  "fixedWeeks",
  "focusedValue",
  "format",
  "parse",
  "placeholder",
  "getRootNode",
  "id",
  "ids",
  "isDateUnavailable",
  "locale",
  "max",
  "min",
  "name",
  "numOfMonths",
  "onFocusChange",
  "onOpenChange",
  "onValueChange",
  "onViewChange",
  "open",
  "defaultOpen",
  "positioning",
  "readOnly",
  "selectionMode",
  "startOfWeek",
  "timeZone",
  "translations",
  "value",
  "defaultView",
  "defaultValue",
  "view",
  "defaultFocusedValue",
  "minView",
  "maxView"
]);
var splitProps = utils.createSplitProps(props);
var inputProps = types.createProps()(["index", "fixOnBlur"]);
var splitInputProps = utils.createSplitProps(inputProps);
var presetTriggerProps = types.createProps()(["value"]);
var splitPresetTriggerProps = utils.createSplitProps(presetTriggerProps);
var tableProps = types.createProps()(["columns", "id", "view"]);
var splitTableProps = utils.createSplitProps(tableProps);
var tableCellProps = types.createProps()(["disabled", "value", "columns"]);
var splitTableCellProps = utils.createSplitProps(tableCellProps);
var viewProps = types.createProps()(["view"]);
var splitViewProps = utils.createSplitProps(viewProps);

exports.anatomy = anatomy;
exports.connect = connect;
exports.inputProps = inputProps;
exports.machine = machine;
exports.parse = parse;
exports.presetTriggerProps = presetTriggerProps;
exports.props = props;
exports.splitInputProps = splitInputProps;
exports.splitPresetTriggerProps = splitPresetTriggerProps;
exports.splitProps = splitProps;
exports.splitTableCellProps = splitTableCellProps;
exports.splitTableProps = splitTableProps;
exports.splitViewProps = splitViewProps;
exports.tableCellProps = tableCellProps;
exports.tableProps = tableProps;
exports.viewProps = viewProps;
